import unittest
from . import oidc_tcpserver
import base64
from ..generic import generate_random_string, json_encode, to_bytes, to_ustring
from .. import oidc


class TestOIDC(unittest.TestCase):
    @classmethod
    def setUpClass(self):
        print('Test against local http(s) server')
        # start a static http server which serves a jwt
        self.host = 'localhost'
        self.port = 4711
        self.server, \
            self.server_thread = oidc_tcpserver.startDynamicServer(
                host=self.host,
                port=self.port,
                use_ssl=True,
            )

    @classmethod
    def tearDownClass(self):
        # stop the server again
        print('Stopping local http(s) server')
        oidc_tcpserver.stopServer(self.server, self.server_thread)
        print('Test against local server done!')

    @unittest.skip("Fails in line 94 in oidc.getReq()")
    def test_oidc(self):
        print('>>> Running oidc authentication tests')
        server_base = 'https://%s:%s' % (self.host, self.port)
        auth_uri_path = '/oidc_auth'
        auth_uri = server_base + auth_uri_path
        redirect_uri = 'https://foo.example.com'
        token_uri_path = '/oidc_token'
        token_uri = server_base + token_uri_path
        userinfo_uri_path = '/oidc_userinfo'
        userinfo_uri = server_base + userinfo_uri_path
        state = generate_random_string(mode='normal', length=128)

        client_id = 'SomeRandomValue'
        client_secret = (
            'wFI1rznlwiApOhmRT2iD5m40nGTjXmPYEMQZ3Ie61pJo'
            'rME8ptBM52'
        )

        # do not verify SSL certificates
        verify_ssl = False

        # try to check the signature of JWTs
        verify_jwt = True

        # 1.)
        # prepare the server and create an URL to access it
        # set the expected parameters for our first call
        expected_params = {
            'redirect_uri': redirect_uri,
            'state': state,
            'client_id': client_id
        }
        self.server.RequestHandlerClass.expected_params = expected_params
        expected_urls = [auth_uri_path]
        self.server.RequestHandlerClass.expected_urls = expected_urls

        # start login process, redirect the user to the OP
        target_url, rec_state = oidc.authRedir(
            auth_uri=auth_uri,
            redirect_uri=redirect_uri,
            client_id=client_id,
            state=state
        )
        self.assertTrue(target_url.startswith(auth_uri))
        self.assertEqual(rec_state, state)

        # 2.)
        # the user is redirected to the OP login page
        # user logs into OpenID-Connect backend server (OP)
        # the redirect page needs to invoke a callback
        # we fake this here and set code and state ourself!

        # edit the expected parameters on the server side
        expected_params['scope'] = 'openid'
        expected_params['response_type'] = 'code'
        self.server.RequestHandlerClass.expected_params = expected_params
        code = '1234567'
        answer = {
            'code': code,
            'state': state
        }
        self.server.RequestHandlerClass.answer = answer

        print('>>> Request to OP for user login')
        res = oidc.getReq(url=target_url, verify=verify_ssl)
        assert res.status_code == 200, 'Server gave an error: %s' % res.text
        print(res.status_code)
        print('DEBUG: content: %s' % res.content)
        print('DEBUG: text: %s' % res.text)
        resdict = eval(res.text)
        # we expect a 'code' and the previously given 'state'
        assert resdict['state'] == state, (
            'State differs! Received: %s '
            'Expected: %s' % (resdict['state'], state)
        )
        assert resdict['code'] == code, (
            'Code differs! Received: %s '
            'Expected: %s' % (resdict['code'], code)
        )

        # 3.)
        # prepare the server
        # allowed expected_urls
        expected_urls = [token_uri_path]
        self.server.RequestHandlerClass.expected_urls = expected_urls

        # expected parameters
        expected_params['code'] = code
        expected_params['grant_type'] = 'authorization_code'
        self.server.RequestHandlerClass.expected_params = expected_params

        # allowed headers
        basic = base64.b64encode(to_bytes(client_id + ':' + client_secret))
        expected_headers = {'Authorization': 'Basic ' + to_ustring(basic)}
        self.server.RequestHandlerClass.expected_headers.update(
            expected_headers
        )

        # preconfigured answer
        jwt_key = '9348235'
        jwt_token = {
            "iss": server_base,
            "sub": "testuser",
            "aud": client_id,
            "nonce": "xjs8Yu0-2",
            "exp": 2147483646,
            "iat": 507212328,
        }
        jwt = oidc.createJwt(
            id_token=jwt_token,
            key=jwt_key,
            alg='HS256'
        )
        access_token = 'asdfhasdfsdfljsadf3434sadlfasdf'
        answer = {
            'id_token': jwt,
            'access_token': access_token,
        }
        answer = json_encode(answer)
        self.server.RequestHandlerClass.answer = answer

        # we make an request to the OP over our backchannel connection
        print('>>>  request to OP to receive login details')
        res = oidc.authReq(
            url=token_uri,
            auth_code=code,
            client_id=client_id,
            client_secret=client_secret,
            redirect_uri=redirect_uri,
            verify=verify_ssl
        )
        assert res['status_code'] == 200, 'Server gave an error: %s' % res.text

        # parse the received data (JWT) and extract useful information
        res = oidc.parseAuthResp(
            res,
            aud=client_id,
            key=jwt_key,
            verify=verify_jwt,
        )
        assert res['id_token'] == jwt_token, 'JWT was altered!'
        assert res['access_token'] == access_token, (
            'Acess token does not match'
        )

        # 4.) Request additional user information using the access_token
        expected_headers = {'Authorization':  'Bearer ' + access_token}
        self.server.RequestHandlerClass.expected_headers.update(
            expected_headers
        )

        print('>>> Request user info from the OP')
        answer = {'foo': 'bar'}
        answer = json_encode(answer)
        self.server.RequestHandlerClass.answer = answer
        # request additional information about the user (email, names,
        # groups...)
        res = oidc.userInfoReq(
            url=userinfo_uri,
            access_token=access_token,
            verify=verify_ssl
        )
        assert res['status_code'] == 200, (
            'Server error while requesting user info'
        )
        print(res['text'])
